<?php
/**
 * Headless Dependency Manager
 *
 * Monitors, installs, and activates required/optional plugins & themes.
 */

add_action('admin_menu', function () {
    add_plugins_page(
        'Headless Dependencies',    // Page title
        'Headless Deps',            // Menu title (shows under "Plugins")
        'manage_options',           // Capability
        'headless-deps',            // Menu slug
        'headless_deps_page'        // Callback
    );
});
/**
 * Load deps array from external file (deps.php)
 */
function headless_deps_get_config()
{
    $config = include plugin_dir_path(__FILE__) . 'headless-deps.php';
    return $config ?: [];
}

/**
 * Render dependencies page
 */
function headless_deps_page()
{
    $deps = headless_deps_get_config();
    echo '<div class="wrap"><h1>Headless Dependencies</h1>';

    // Themes
    if (!empty($deps['themes'])) {
        echo '<h2>Themes</h2>';
        headless_deps_render_table($deps['themes'], 'theme');
    }

    // Plugins
    if (!empty($deps['plugins'])) {
        echo '<h2>Plugins</h2>';
        headless_deps_render_table($deps['plugins'], 'plugin');
    }

    echo '</div>';
}

/**
 * Render dependency table
 */
function headless_deps_render_table($items, $type = 'plugin')
{
    echo '<table class="widefat striped">';
    echo '<thead><tr>
            <th>Name</th>
            <th>Version</th>
            <th>Author</th>
            <th>Required?</th>
            <th>Status</th>
            <th>Update Policy</th>
            <th>Action</th>
          </tr></thead><tbody>';

    foreach ($items as $item) {
        headless_deps_render_row($item, $type);
        if (!empty($item['children'])) {
            foreach ($item['children'] as $child) {
                headless_deps_render_row($child, $type, true);
            }
        }
    }

    echo '</tbody></table>';
}

/**
 * Render a single row
 */
function headless_deps_render_row($item, $type, $is_child = false)
{
    $slug = $item['slug'] ?? '';
    $name = ($is_child ? '— ' : '') . esc_html($item['name']);
    $version = esc_html($item['version'] ?? '');
    $author = esc_html($item['author'] ?? '');
    $required = !empty($item['required']) ? 'Yes' : 'No';
    $allow_update = isset($item['allow_update']) && $item['allow_update'] ? 'Allowed' : 'Locked';
    $url = esc_url($item['url'] ?? '');

    // Detect install/active status
    $status = 'Not Installed';
    $action = '';
    if ($type === 'plugin') {
        if (file_exists(WP_PLUGIN_DIR . "/$slug")) {
            if (is_plugin_active("$slug/$slug.php")) {
                $status = 'Active';
                $action = '<a href="' . admin_url('plugins.php') . '" class="button">Manage</a>';
            } else {
                $status = 'Installed (Inactive)';
                $action = '<a href="' . admin_url('plugins.php') . '" class="button">Activate</a>';
            }
        } else {
            $status = 'Not Installed';
            $action = '<a href="' . $url . '" class="button button-primary" target="_blank">Download</a>';
        }
    } elseif ($type === 'theme') {
        if (wp_get_theme($slug)->exists()) {
            if (wp_get_theme()->get_stylesheet() === $slug) {
                $status = 'Active';
                $action = '<a href="' . admin_url('themes.php') . '" class="button">Manage</a>';
            } else {
                $status = 'Installed (Inactive)';
                $action = '<a href="' . admin_url('themes.php') . '" class="button">Activate</a>';
            }
        } else {
            $status = 'Not Installed';
            $action = '<a href="' . $url . '" class="button button-primary" target="_blank">Download</a>';
        }
    }

    // Build row
    echo '<tr>
            <td>' . $name . '</td>
            <td>' . $version . '</td>
            <td>' . $author . '</td>
            <td>' . $required . '</td>
            <td>' . $status . '</td>
            <td>' . $allow_update . '</td>
            <td>' . $action . '</td>
          </tr>';
}
